/**
* \file: am_api_fsm.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "am_api_fsm.h"

//shared library incs
#include "am_api_dispatcher.h"
#include "am_api_socket_client.h"
#include "am_api_events.h"
#include "am_api_control.h"

//automounter_shared incs
#include "utils/logger.h"
#include "utils/version_info.h"
#include "ipc/message_sendr.h"
#include "ipc/message_recvr.h"

//------------------------------------- private attributes ------------------------------------------------------------
static automounter_api_state_t state = AM_API_UNINITIALIZED;
static const char* am_api_fsm_app_identifier;
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ----------------------------------------------------
static void am_api_fsm_enter_uninitialized(void);
static void am_api_fsm_enter_connecting(void);
static void am_api_fsm_enter_connected(void);
static void am_api_fsm_enter_disconnected(void);
static error_code_t am_api_fsm_enter_waiting_for_daemon(void);

static error_code_t am_api_fsm_check_state_before_connect(void);
static error_code_t am_api_fsm_do_connect(bool send_signals);

//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member definition ------------------------------------------------------
error_code_t am_api_fsm_signal_init_request(const char *app_identifer,
		logger_loglevel_t loglevel, bool console_log_enabled)
{
	error_code_t result;

	if (state!=AM_API_UNINITIALIZED)
	{
		logger_log_info("AUTOMOUNTER_API - Shared library already initialized. Ignoring the 2nd init request.");
		return RESULT_OK;
	}

	am_api_fsm_app_identifier=app_identifer;
	logger_init(app_identifer,loglevel,console_log_enabled);
	logger_log_debug("AUTOMOUNTER_API - Shared library initialized.");
	logger_log_info("AUTOMOUNTER_API - Shared library version: %s",VERSION_INFO_FORMATED_LINE);

	am_api_events_register_callbacks(NULL);

	result=am_api_socket_client_init();
	if (result==RESULT_OK)
		result=am_api_dispatcher_init();

	if (result==RESULT_OK)
		am_api_fsm_enter_disconnected();
	return result;
}

void am_api_fsm_signal_deinit_request(void)
{
	if (state==AM_API_UNINITIALIZED)
	{
		logger_log_info("AUTOMOUNTER_API - Shared library not initialized. Ignoring the deinit request.");
		return;
	}

	am_api_events_register_callbacks(NULL);

	if (state==AM_API_INITIALIZED_CONNECTED)
		am_api_socket_client_disconnect();
	am_api_dispatcher_deinit();
	am_api_socket_client_deinit();

	am_api_fsm_enter_uninitialized();
}

error_code_t am_api_fsm_signal_connect_request(void)
{
	error_code_t result;
	result=am_api_fsm_check_state_before_connect();

	//do nothing in case we are in an invalid state, or we are connecting, or we are already connected
	if (result != RESULT_OK || state==AM_API_INITIALIZED_CONNECTED || state==AM_API_INITIALIZED_CONNECTING)
		return result;

	return am_api_fsm_do_connect(false);
}

error_code_t am_api_fsm_signal_try_connect_request(void)
{
	error_code_t result;
	result=am_api_fsm_check_state_before_connect();

	//do nothing in case we are in an invalid state, or we are connecting, or we are already connected
	if (result != RESULT_OK || state==AM_API_INITIALIZED_CONNECTED || state==AM_API_INITIALIZED_CONNECTING)
		return result;

	result=am_api_socket_client_check_daemon_running();

	if (result==RESULT_DAEMON_NOT_RUNNING)
		//automounter not running, go into wait state
		return am_api_fsm_enter_waiting_for_daemon();

	//automounter running, connect with it
	if (result==RESULT_OK)
		result=am_api_fsm_do_connect(true);

	return result;
}

void am_api_fsm_signal_disconnect_request(void)
{
	if (state==AM_API_UNINITIALIZED || state == AM_API_INITIALIZED_DISCONNECTED)
		return;

	if (state==AM_API_INITIALIZED_WAITING_FOR_DAEMON)
		am_api_socket_client_stop_waiting_for_socket();

	if (state==AM_API_INITIALIZED_CONNECTED)
		am_api_socket_client_disconnect();

	//we don't need to check for AM_API_CONNECTING since no one is able to call this function in this state.
	//Thread locks are preventing this.
	am_api_fsm_enter_disconnected();
}

void am_api_fsm_signal_connection_lost(void)
{
	if (state!=AM_API_INITIALIZED_CONNECTED)
		return;

	am_api_socket_client_disconnect();
	am_api_fsm_enter_disconnected();
	am_api_dispatcher_signal_connection_lost();
}

void am_api_fsm_signal_automounter_appeared(void)
{
	if (am_api_fsm_do_connect(true)!=RESULT_OK)
		logger_log_error("AUTOMOUNTER_API - Error connecting with the automounter automatically after it appeared.");
}

automounter_api_state_t am_api_fsm_get_state(void)
{
	return state;
}

void am_api_fsm_signal_callbacks_changed(void)
{
	if (state==AM_API_INITIALIZED_CONNECTED)
	{
		event_mask_t event_mask=am_api_events_get_callback_mask();
		am_api_control_send_application_info(am_api_fsm_app_identifier,event_mask);
	}
}
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ----------------------------------------------------
static void am_api_fsm_enter_uninitialized(void)
{
	logger_deinit();
	state=AM_API_UNINITIALIZED;
	logger_log_debug("AUTOMOUNTER_API - Shared library deinitialized.");
}

static void am_api_fsm_enter_connecting(void)
{
	state=AM_API_INITIALIZED_CONNECTING;
	logger_log_debug("AUTOMOUNTER_API - Shared library entered state CONNECTING.");
}

static void am_api_fsm_enter_connected(void)
{
	event_mask_t event_mask;
	state=AM_API_INITIALIZED_CONNECTED;
	logger_log_debug("AUTOMOUNTER_API - Shared library entered state CONNECTED.");
	event_mask=am_api_events_get_callback_mask();
	am_api_control_send_application_info(am_api_fsm_app_identifier,event_mask);
}

static void am_api_fsm_enter_disconnected(void)
{
	state=AM_API_INITIALIZED_DISCONNECTED;
	logger_log_debug("AUTOMOUNTER_API - Shared library entered state DISCONNECTED.");
}

static error_code_t am_api_fsm_enter_waiting_for_daemon(void)
{
	error_code_t result;

	//setting state already here because the library might enter state CONNECTED while calling start_waiting_for_socket()
	//this happens in case the automounter appears in the meanwhile
	state=AM_API_INITIALIZED_WAITING_FOR_DAEMON;

	result=am_api_socket_client_start_waiting_for_socket();
	if (result!=RESULT_OK)
	{
		//in case of an error, we have to reset the socket and go back to state DISCONNECTED
		am_api_socket_client_stop_waiting_for_socket();
		state=AM_API_INITIALIZED_DISCONNECTED;
	}

	return result;
}

static error_code_t am_api_fsm_check_state_before_connect(void)
{
	if (state==AM_API_UNINITIALIZED)
		return RESULT_INVALID;

	if (state==AM_API_INITIALIZED_CONNECTED)
	{
		logger_log_info("AUTOMOUNTER_API - Shared library already connected. Ignoring the connect request.");
		return RESULT_OK;
	}

	if (state==AM_API_INITIALIZED_CONNECTING)
	{
		logger_log_info("AUTOMOUNTER_API - Shared library already establishing a connecting. Ignoring the connect request.");
		return RESULT_OK;
	}

	return RESULT_OK;
}

static error_code_t am_api_fsm_do_connect(bool send_signals)
{
	error_code_t result;

	am_api_fsm_enter_connecting();
	result=am_api_socket_client_connect();
	if (result==RESULT_OK)
	{
		am_api_fsm_enter_connected();
		if (send_signals)
			am_api_dispatcher_signal_establish_connection_success();
	}
	else
	{
		am_api_fsm_enter_disconnected();
		if (send_signals)
			am_api_dispatcher_signal_establish_connection_failure();
	}

	return result;
}
//---------------------------------------------------------------------------------------------------------------------
